//
//  Home.swift
//  Do It
//
//  Created by Jim Dovey on 1/27/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI
import Combine

enum HomeSelection: Hashable {
    case list(UUID)
    case group(TodoItemGroup)
}

struct Home: View {
    @EnvironmentObject var data: DataCenter
    
    static private let listTemplate = TodoItemList(name: "New List", color: .blue, icon: "list.bullet")
    @State private var showingEditor = false
    
    @State var selection: HomeSelection?
    @Environment(\.errorPublisher) var errorPublisher

    var body: some View {
        return NavigationView {
            List {
                Section(header: HomeHeader(selection: $selection).padding(.vertical)) {
                    ForEach(data.todoLists) { list in
                        NavigationLink(destination: TodoList(list: list).font(nil),
                                       tag: .list(list.id),
                                       selection: self.$selection) {
                            Row(name: list.name,icon: list.icon, color: list.color.uiColor)
                        }
                    }
                    .onDelete { self.data.removeLists(atOffsets: $0) }
                    .onMove { self.data.moveLists(fromOffsets: $0, to: $1) }
                }
            }
            .font(.system(.headline, design: .rounded))
            .listStyle(GroupedListStyle())
            .navigationBarTitle("Lists")
            .navigationBarItems(
                leading: EditButton(),
                trailing: addButton)
        }
        .sheet(isPresented: $showingEditor) {
            TodoListEditor(list: Self.listTemplate)
        }
    }
    
    private var addButton: some View {
        Button(action: { self.showingEditor.toggle() }) {
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New List"))
        }
    }

    private struct Row: View {
        var name: String
        var icon: String
        var color: Color

        var body: some View {
            HStack {
                Image(systemName: icon)
                    .foregroundColor(.white)
                    .frame(width: 32, height: 32)
                    .background(color)
                    .clipShape(Circle())
                Text(LocalizedStringKey(name))
            }
        }
    }
}

struct TodoListChooser_Previews: PreviewProvider {
    static var previews: some View {
        Home()
            .environmentObject(DataCenter())
            .padding(.top, 45)
    }
}
